const fs = require('fs');
const path = require('path');
const { Resvg } = require('@resvg/resvg-js');

const projectRoot = path.resolve(__dirname, '..');
const assetsDir = path.join(projectRoot, 'assets');
const buildDir = path.join(projectRoot, 'build');

const svgPath = path.join(assetsDir, 'app-icon.svg');
const pngPath = path.join(assetsDir, 'app-icon.png');
const logoSourcePath = path.join(assetsDir, 'logo-source', 'video-manager-logo.png');
const icoPath = path.join(buildDir, 'icon.ico');
const assetsIcoPath = path.join(assetsDir, 'app-icon.ico');
const buildPngPath = path.join(buildDir, 'icon.png');
const icnsPath = path.join(buildDir, 'icon.icns');
const assetsIcnsPath = path.join(assetsDir, 'app-icon.icns');

function ensureAssetsDir() {
  if (!fs.existsSync(assetsDir)) {
    fs.mkdirSync(assetsDir, { recursive: true });
  }
}

function ensureBuildDir() {
  if (!fs.existsSync(buildDir)) {
    fs.mkdirSync(buildDir, { recursive: true });
  }
}

function renderSvgToPng() {
  if (!fs.existsSync(svgPath)) {
    return null;
  }

  const svgContent = fs.readFileSync(svgPath, 'utf8');
  const resvg = new Resvg(svgContent, {
    background: 'rgba(0,0,0,0)',
    fitTo: {
      mode: 'width',
      value: 512
    }
  });

  const pngData = resvg.render();
  const pngBuffer = pngData.asPng();
  try {
    fs.writeFileSync(pngPath, pngBuffer);
  } catch (error) {
    if (error && (error.code === 'EBUSY' || error.code === 'EPERM')) {
      console.warn(`Warning: Could not update ${pngPath} because it is currently in use. The previous file will be kept.`);
    } else {
      throw error;
    }
  }
  fs.writeFileSync(buildPngPath, pngBuffer);
  return pngBuffer;
}

async function generateMultiSizePngBuffers(sourcePath, iconSizes) {
  const { default: Jimp } = await import('jimp');
  const baseImage = await Jimp.read(sourcePath);
  return await Promise.all(
    iconSizes.map(async size => {
      const resized = baseImage.clone().resize(size, size, Jimp.RESIZE_BICUBIC);
      return resized.getBufferAsync(Jimp.MIME_PNG);
    })
  );
}

async function generateIco({ useSvgSource, sourcePngPath }) {
  const { default: pngToIco } = await import('png-to-ico');
  // NSIS requires icon entries to be 256px or smaller. Generate only supported sizes.
  const iconSizes = [256, 128, 64, 48, 32, 24, 16];
  let multiSizePngBuffers;

  if (useSvgSource && fs.existsSync(svgPath)) {
    multiSizePngBuffers = iconSizes.map(size => {
      const resvg = new Resvg(fs.readFileSync(svgPath, 'utf8'), {
        background: 'rgba(0,0,0,0)',
        fitTo: {
          mode: 'width',
          value: size
        }
      });
      return resvg.render().asPng();
    });
  } else {
    if (!fs.existsSync(sourcePngPath)) {
      throw new Error('Cannot generate ICO: no SVG or PNG source available.');
    }
    multiSizePngBuffers = await generateMultiSizePngBuffers(sourcePngPath, iconSizes);
  }

  const icoBuffer = await pngToIco(multiSizePngBuffers);
  fs.writeFileSync(icoPath, icoBuffer);
  fs.writeFileSync(assetsIcoPath, icoBuffer);
}

async function generateIcns(sourcePngPath) {
  const iconGen = require('icon-gen');
  
  try {
    // icon-gen creates files based on the source filename or uses 'app' as default
    // We'll generate it and then rename to icon.icns
    const tempIcnsPath = path.join(buildDir, 'app.icns');
    
    await iconGen(sourcePngPath, buildDir, {
      modes: ['icns'],
      report: false
    });
    
    // icon-gen creates app.icns by default, rename it to icon.icns
    if (fs.existsSync(tempIcnsPath)) {
      // Remove old icon.icns if it exists
      if (fs.existsSync(icnsPath)) {
        fs.unlinkSync(icnsPath);
      }
      fs.renameSync(tempIcnsPath, icnsPath);
      console.log(`Saved ICNS icon to ${icnsPath}`);
    }
    
    // Copy to assets directory if build was successful
    if (fs.existsSync(icnsPath)) {
      if (fs.existsSync(assetsIcnsPath)) {
        fs.unlinkSync(assetsIcnsPath);
      }
      fs.copyFileSync(icnsPath, assetsIcnsPath);
    }
  } catch (error) {
    console.warn('Failed to generate ICNS file:', error.message || error);
    // ICNS generation is optional, don't fail the entire process
  }
}

async function main() {
  ensureBuildDir();
  ensureAssetsDir();

  let pngBuffer = null;
  let sourcePngPath = null;
  const svgExists = fs.existsSync(svgPath);
  const logoSourceExists = fs.existsSync(logoSourcePath);

  // Priority: logo-source PNG > SVG > existing PNG
  if (logoSourceExists) {
    console.log(`Using logo source from ${logoSourcePath}...`);
    pngBuffer = fs.readFileSync(logoSourcePath);
    sourcePngPath = logoSourcePath;
    // Copy to assets/app-icon.png as the primary source
    fs.writeFileSync(pngPath, pngBuffer);
    fs.writeFileSync(buildPngPath, pngBuffer);
    console.log(`Saved PNG icon to ${pngPath}`);
  } else if (svgExists) {
    console.log('Rendering SVG to PNG...');
    pngBuffer = renderSvgToPng();
    sourcePngPath = pngPath;
    console.log(`Saved PNG icon to ${pngPath}`);
  } else if (fs.existsSync(pngPath)) {
    console.warn(`SVG icon not found at ${svgPath}. Using existing PNG at ${pngPath}.`);
    pngBuffer = fs.readFileSync(pngPath);
    sourcePngPath = pngPath;
    fs.writeFileSync(buildPngPath, pngBuffer);
  } else if (fs.existsSync(buildPngPath)) {
    console.warn(`SVG icon not found at ${svgPath}. Duplicating existing build PNG to assets.`);
    pngBuffer = fs.readFileSync(buildPngPath);
    sourcePngPath = buildPngPath;
    fs.writeFileSync(pngPath, pngBuffer);
  } else {
    console.error(`No icon source found. Expected ${svgPath}, ${pngPath}, or ${logoSourcePath}.`);
    process.exit(1);
  }

  console.log('Generating ICO variants...');
  await generateIco({ useSvgSource: svgExists && !logoSourceExists, sourcePngPath });
  console.log(`Saved ICO icon to ${icoPath}`);

  console.log('Generating ICNS variants...');
  await generateIcns(sourcePngPath);
}

main().catch(error => {
  console.error('Failed to generate icons:', error);
  process.exit(1);
});

